// Modern ReviewPR JavaScript
class ReviewPR {
    constructor() {
        this.products = [];
        this.currentProduct = null;
        this.performanceMetrics = {
            startTime: performance.now(),
            loadTime: null,
            errors: []
        };
        this.init();
    }

    init() {
        this.setupEventListeners();
        // Enable product loading - load from database for Read More functionality
        this.loadProducts();
        this.initScrollEffects();
        this.initAnimations();
        this.initQuickOrderForm();
        
        // Hide loading div after products are loaded
        const loading = document.getElementById('loading');
        if (loading) {
            // Set a timeout to hide loading in case products load quickly
            setTimeout(() => {
                loading.style.display = 'none';
            }, 1000);
        }
    }

    setupEventListeners() {
        // Smooth scrolling for navigation links
        document.querySelectorAll('a[href^="#"]').forEach(anchor => {
            anchor.addEventListener('click', (e) => {
                e.preventDefault();
                const target = document.querySelector(anchor.getAttribute('href'));
                if (target) {
                    target.scrollIntoView({
                        behavior: 'smooth',
                        block: 'start'
                    });
                }
            });
        });

        // Mobile menu toggle
        const mobileMenuBtn = document.querySelector('.mobile-menu-btn');
        const mobileMenu = document.querySelector('.mobile-menu');
        
        if (mobileMenuBtn && mobileMenu) {
            mobileMenuBtn.addEventListener('click', () => {
                mobileMenu.classList.toggle('hidden');
            });
        }

        // Quick order form
        const quickOrderForm = document.getElementById('quick-order-form');
        if (quickOrderForm) {
            quickOrderForm.addEventListener('submit', (e) => {
                e.preventDefault();
                this.handleQuickOrder(new FormData(quickOrderForm));
            });
        }
    }

    initScrollEffects() {
        // Navbar scroll effect
        const navbar = document.querySelector('.navbar');
        if (navbar) {
            window.addEventListener('scroll', () => {
                if (window.scrollY > 50) {
                    navbar.classList.add('scrolled');
                } else {
                    navbar.classList.remove('scrolled');
                }
            });
        }

        // Intersection Observer for animations
        const observerOptions = {
            threshold: 0.1,
            rootMargin: '0px 0px -50px 0px'
        };

        const observer = new IntersectionObserver((entries) => {
            entries.forEach(entry => {
                if (entry.isIntersecting) {
                    entry.target.classList.add('fade-in');
                }
            });
        }, observerOptions);

        // Observe elements for animation
        document.querySelectorAll('.service-card, .feature-item, .testimonial-card, .product-card').forEach(el => {
            observer.observe(el);
        });
    }

    initAnimations() {
        // Counter animation for stats
        this.animateCounters();
        
        // Typing effect for hero tagline
        this.initTypingEffect();
    }

    initQuickOrderForm() {
        // Initialize Quick Order form with real product data
        const platformSelect = document.getElementById('quick-platform');
        const priceDisplay = document.getElementById('price-display');
        const selectedPrice = document.getElementById('selected-price');
        const selectedDescription = document.getElementById('selected-description');
        const orderBtn = document.getElementById('quick-order-btn');

        if (!platformSelect) return;

        // Platform selection handler
        platformSelect.addEventListener('change', (e) => {
            const selectedOption = e.target.selectedOptions[0];
            const selectedProductId = e.target.value;
            
            if (selectedProductId) {
                // Check if it's a temporary option (has data-price attribute)
                if (selectedOption.hasAttribute('data-price')) {
                    const price = selectedOption.getAttribute('data-price');
                    const serviceName = selectedOption.textContent.split(' - $')[0];
                    
                    selectedPrice.textContent = `$${parseFloat(price).toFixed(2)}`;
                    selectedDescription.textContent = `Professional ${serviceName.toLowerCase()} service`;
                    priceDisplay.style.display = 'block';
                    orderBtn.disabled = false;
                    orderBtn.innerHTML = '<i class="fas fa-shopping-cart mr-2"></i>Order Now - $' + parseFloat(price).toFixed(2);
                }
                // Check if it's a real product from database
                else if (this.products.length > 0) {
                    const product = this.products.find(p => p.id == selectedProductId);
                    
                    if (product) {
                        selectedPrice.textContent = `$${parseFloat(product.price).toFixed(2)}`;
                        selectedDescription.textContent = product.description || product.name;
                        priceDisplay.style.display = 'block';
                        orderBtn.disabled = false;
                        orderBtn.innerHTML = '<i class="fas fa-shopping-cart mr-2"></i>Order Now - $' + parseFloat(product.price).toFixed(2);
                    } else {
                        this.resetQuickOrderForm();
                    }
                }
            } else {
                this.resetQuickOrderForm();
            }
        });
    }

    resetQuickOrderForm() {
        const priceDisplay = document.getElementById('price-display');
        const orderBtn = document.getElementById('quick-order-btn');
        
        if (priceDisplay) priceDisplay.style.display = 'none';
        if (orderBtn) {
            orderBtn.disabled = true;
            orderBtn.innerHTML = '<i class="fas fa-shopping-cart mr-2"></i>Select Service First';
        }
    }

    populateQuickOrderForm() {
        const platformSelect = document.getElementById('quick-platform');
        
        if (!platformSelect) return;

        // If no products loaded, keep the temporary options
        if (this.products.length === 0) {
            console.log('No products loaded, keeping temporary options');
            return;
        }

        // Clear existing options and add real products
        platformSelect.innerHTML = '<option value="">Choose a service...</option>';

        // Add real products as options
        this.products.forEach(product => {
            const option = document.createElement('option');
            option.value = product.id;
            option.textContent = `${product.name} - $${parseFloat(product.price).toFixed(2)}`;
            platformSelect.appendChild(option);
        });

        console.log(`Populated Quick Order form with ${this.products.length} real products`);
    }

    populateFeaturedServices() {
        const featuredContainer = document.getElementById('featured-services');
        
        if (!featuredContainer || this.products.length === 0) return;

        // Get first 3 products as featured services
        const featuredProducts = this.products.slice(0, 3);
        
        // Icon mapping for different service types
        const iconMap = {
            'trustpilot': 'fab fa-trustpilot',
            'google': 'fab fa-google',
            'g2': 'fas fa-chart-line',
            'capterra': 'fas fa-star',
            'default': 'fas fa-star'
        };

        featuredContainer.innerHTML = '';

        featuredProducts.forEach(product => {
            // Determine icon based on product name/category
            let iconClass = iconMap.default;
            const productName = product.name.toLowerCase();
            
            if (productName.includes('trustpilot')) iconClass = iconMap.trustpilot;
            else if (productName.includes('google')) iconClass = iconMap.google;
            else if (productName.includes('g2')) iconClass = iconMap.g2;
            else if (productName.includes('capterra')) iconClass = iconMap.capterra;

            const serviceCard = document.createElement('div');
            serviceCard.className = 'bg-white rounded-xl shadow-lg hover-lift p-8 border border-gray-100';
            
            serviceCard.innerHTML = `
                <div class="text-4xl feature-icon mb-4">
                    <i class="${iconClass}"></i>
                </div>
                <h3 class="text-xl font-bold text-gray-900 mb-3">${product.name}</h3>
                <p class="text-gray-600 mb-4">${product.description || 'Professional review service to boost your online reputation.'}</p>
                <div class="text-2xl font-bold text-gradient mb-4">$${parseFloat(product.price).toFixed(2)}</div>
                <button onclick="reviewPR.orderProduct('${product.id}')" class="w-full bg-blue-50 text-blue-600 py-2 rounded-lg font-semibold hover:bg-blue-100 transition-colors">
                    Order Now
                </button>
            `;
            
            featuredContainer.appendChild(serviceCard);
        });
    }

    animateCounters() {
        const counters = document.querySelectorAll('.stats-counter');
        
        const animateCounter = (counter) => {
            const target = parseInt(counter.dataset.count || counter.textContent.replace(/[^\d.]/g, ''));
            const duration = 2000;
            const start = performance.now();
            
            const update = (currentTime) => {
                const elapsed = currentTime - start;
                const progress = Math.min(elapsed / duration, 1);
                
                const current = Math.floor(progress * target);
                counter.textContent = this.formatNumber(current);
                
                if (progress < 1) {
                    requestAnimationFrame(update);
                }
            };
            
            requestAnimationFrame(update);
        };

        const observer = new IntersectionObserver((entries) => {
            entries.forEach(entry => {
                if (entry.isIntersecting) {
                    animateCounter(entry.target);
                    observer.unobserve(entry.target);
                }
            });
        });

        counters.forEach(counter => observer.observe(counter));
    }

    formatNumber(num) {
        if (num >= 1000) {
            return (num / 1000).toFixed(1) + 'K+';
        }
        return num.toString();
    }

    initTypingEffect() {
        const typingElement = document.querySelector('.typing-text');
        if (!typingElement) return;

        const text = typingElement.dataset.text || typingElement.textContent;
        const speed = 100;
        let i = 0;

        typingElement.textContent = '';

        const typeWriter = () => {
            if (i < text.length) {
                typingElement.textContent += text.charAt(i);
                i++;
                setTimeout(typeWriter, speed);
            }
        };

        // Start typing effect when element is visible
        const observer = new IntersectionObserver((entries) => {
            entries.forEach(entry => {
                if (entry.isIntersecting) {
                    typeWriter();
                    observer.unobserve(entry.target);
                }
            });
        });

        observer.observe(typingElement);
    }

    async loadProducts() {
        const container = document.getElementById('products-container');
        const loading = document.getElementById('loading');
        
        // Check if we already have static products loaded
        if (container && container.children.length > 0) {
            console.log('Static products already loaded, skipping database load');
            if (loading) loading.style.display = 'none';
            return;
        }
        
        if (!container) {
            this.logError('Products container not found');
            return;
        }

        try {
            if (loading) loading.style.display = 'block';
            const loadStart = performance.now();
            
            const response = await fetch('products-simple.php', {
                method: 'GET',
                headers: {
                    'X-Requested-With': 'XMLHttpRequest',
                    'Accept': 'application/json',
                    'Accept-Encoding': 'gzip, deflate'
                },
                cache: 'default' // Allow browser caching
            });

            if (!response.ok) {
                throw new Error(`HTTP ${response.status}: ${response.statusText}`);
            }

            const products = await response.json();
            this.products = products;
            
            const loadTime = performance.now() - loadStart;
            this.performanceMetrics.loadTime = loadTime;
            
            loading.style.display = 'none';
            this.displayProducts(products);
            
            // Populate Quick Order form with real product data
            this.populateQuickOrderForm();
            
            // Populate featured services section
            this.populateFeaturedServices();
            
            // Log performance metrics
            console.log(`Products loaded in ${loadTime.toFixed(2)}ms`);
            
        } catch (error) {
            this.logError('Error loading products', error);
            
            // Hide the loading div since we have static products
            if (loading) {
                loading.style.display = 'none';
            }
            
            console.log('Database products failed to load, using static products');
            
            // Don't show error since we have static products working
            // The static products in the HTML will be visible instead
        }
    }

    displayProducts(products) {
        const container = document.getElementById('products-container');
        if (!container) return;

        // Force clear existing content and load from database
        container.innerHTML = '';

        if (products.length > 0) {
            // Use document fragment for better performance
            const fragment = document.createDocumentFragment();
            
            products.forEach(product => {
                const productCard = this.createProductCard(product);
                fragment.appendChild(productCard);
            });
            
            container.appendChild(fragment);
            
            // Initialize lazy loading for images
            this.initLazyLoading();
        } else {
            // Show message if no products
            container.innerHTML = '<p class="text-center text-gray-500 col-span-full py-8">No products available at the moment.</p>';
        }
    }

    createProductCard(product) {
        const card = document.createElement('div');
        card.className = 'bg-white rounded-xl shadow-lg hover-lift p-6 border border-gray-100 product-card';
        
        card.innerHTML = `
            <div class="mb-4">
                <img src="${product.image || 'data:image/svg+xml;charset=UTF-8,%3Csvg width=\"300\" height=\"200\" viewBox=\"0 0 300 200\" fill=\"none\" xmlns=\"http://www.w3.org/2000/svg\"%3E%3Crect width=\"300\" height=\"200\" fill=\"%23F3F4F6\"/%3E%3Cpath d=\"M150 125L175 100H175L150 75H125L175 50H175L150 25H125L75 50L175 100L150 125Z\" fill=\"%23D1D5DB\"/%3E%3C/svg%3E'}" 
                     alt="${product.name}" 
                     class="w-full h-48 object-cover rounded-lg"
                     loading="lazy"
                     onerror="this.src='data:image/svg+xml;charset=UTF-8,%3Csvg width=\"300\" height=\"200\" viewBox=\"0 0 300 200\" fill=\"none\" xmlns=\"http://www.w3.org/2000/svg\"%3E%3Crect width=\"300\" height=\"200\" fill=\"%23F3F4F6\"/%3E%3Cpath d=\"M150 125L175 100H175L150 75H125L175 50H175L150 25H125L75 50L175 100L150 125Z\" fill=\"%23D1D5DB\"/%3E%3C/svg%3E'">
            </div>
            <div class="mb-4">
                <h3 class="text-xl font-bold text-gray-900 mb-2">${product.name}</h3>
                <div class="product-description mb-3">
                    <p class="text-gray-600 description-short" style="display: block;">${(product.description && product.description.length > 100) ? product.description.substring(0, 100) + '...' : (product.description || 'Professional review service to boost your online reputation')}</p>
                    <p class="text-gray-600 description-full" style="display: none;">${product.description || 'Professional review service to boost your online reputation'}</p>
                    ${(product.description && product.description.length > 100) ? `<button class="read-more-btn inline-flex items-center gap-2 text-blue-600 hover:text-blue-800 text-sm font-medium mt-2 px-3 py-1.5 rounded-full border border-blue-200 hover:border-blue-300 bg-blue-50 hover:bg-blue-100 transition-all duration-200" onclick="toggleDescription(this)">
                        <i class="fas fa-chevron-down text-xs transition-transform duration-200" id="read-more-icon-${product.id}"></i>
                        <span>Read More</span>
                    </button>` : ''}
                </div>
                <div class="flex justify-between items-center mb-4">
                    <span class="text-2xl font-bold text-gradient">$${parseFloat(product.price).toFixed(2)}</span>
                    <span class="bg-blue-100 text-blue-800 px-3 py-1 rounded-full text-sm font-medium">${product.category}</span>
                </div>
            </div>
            <div class="flex space-x-2">
                <button onclick="reviewPR.viewProduct('${product.id}')" 
                        class="flex-1 bg-gray-100 text-gray-700 py-2 px-4 rounded-lg font-semibold hover:bg-gray-200 transition-colors">
                    <i class="fas fa-info-circle mr-1"></i>
                    View Details
                </button>
                <button onclick="reviewPR.orderProduct('${product.id}')" 
                        class="flex-1 btn-primary text-white py-2 px-4 rounded-lg font-semibold">
                    <i class="fas fa-shopping-cart mr-1"></i>
                    Order Now
                </button>
            </div>
        `;
        
        return card;
    }

    viewProduct(productId) {
        const product = this.products.find(p => String(p.id) === String(productId));
        if (!product) {
            this.showNotification('Product not found', 'error');
            return;
        }

        this.currentProduct = product;
        this.showProductModal(product);
    }

    showProductModal(product) {
        // Create modal if it doesn't exist
        let modal = document.getElementById('product-modal');
        if (!modal) {
            modal = this.createProductModal();
            document.body.appendChild(modal);
        }

        // Update modal content
        const modalTitle = modal.querySelector('.modal-title');
        const modalBody = modal.querySelector('.modal-body');
        
        modalTitle.textContent = product.name;
        modalBody.innerHTML = `
            <div class="grid md:grid-cols-2 gap-6">
                <div>
                    <img src="${product.image || 'data:image/svg+xml;charset=UTF-8,%3Csvg width=\"400\" height=\"300\" viewBox=\"0 0 400 300\" fill=\"none\" xmlns=\"http://www.w3.org/2000/svg\"%3E%3Crect width=\"400\" height=\"300\" fill=\"%23F3F4F6\"/%3E%3Cpath d=\"M200 187.5L225 150H225L200 112.5H175L225 75H225L200 37.5H175L125 75L225 150L200 187.5Z\" fill=\"%23D1D5DB\"/%3E%3C/svg%3E'}" 
                         alt="${product.name}" 
                         class="w-full h-64 object-cover rounded-lg"
                         loading="lazy"
                         onerror="this.src='data:image/svg+xml;charset=UTF-8,%3Csvg width=\"400\" height=\"300\" viewBox=\"0 0 400 300\" fill=\"none\" xmlns=\"http://www.w3.org/2000/svg\"%3E%3Crect width=\"400\" height=\"300\" fill=\"%23F3F4F6\"/%3E%3Cpath d=\"M200 187.5L225 150H225L200 112.5H175L225 75H225L200 37.5H175L125 75L225 150L200 187.5Z\" fill=\"%23D1D5DB\"/%3E%3C/svg%3E'">
                </div>
                <div>
                    <h3 class="text-2xl font-bold text-gray-900 mb-3">${product.name}</h3>
                    <p class="text-3xl font-bold text-gradient mb-4">$${parseFloat(product.price).toFixed(2)}</p>
                    <span class="bg-blue-100 text-blue-800 px-3 py-1 rounded-full text-sm font-medium mb-4 inline-block">${product.category}</span>
                    <p class="text-gray-600 mb-6">${product.description || 'Professional review service'}</p>
                    <div class="space-y-3">
                        <div class="flex items-center text-green-600">
                            <i class="fas fa-check-circle mr-2"></i>
                            <span>100% Authentic Reviews</span>
                        </div>
                        <div class="flex items-center text-green-600">
                            <i class="fas fa-check-circle mr-2"></i>
                            <span>24-72 Hour Delivery</span>
                        </div>
                        <div class="flex items-center text-green-600">
                            <i class="fas fa-check-circle mr-2"></i>
                            <span>Money Back Guarantee</span>
                        </div>
                    </div>
                </div>
            </div>
            <div class="mt-6 flex space-x-4">
                <button onclick="reviewPR.orderProduct('${product.id}')" 
                        class="btn-primary text-white px-6 py-3 rounded-lg font-semibold flex-1">
                    <i class="fas fa-shopping-cart mr-2"></i>
                    Order Now - $${parseFloat(product.price).toFixed(2)}
                </button>
                <button onclick="reviewPR.closeModal()" 
                        class="bg-gray-200 text-gray-700 px-6 py-3 rounded-lg font-semibold">
                    Close
                </button>
            </div>
        `;

        // Show modal
        modal.classList.remove('hidden');
        document.body.style.overflow = 'hidden';
    }

    createProductModal() {
        const modal = document.createElement('div');
        modal.id = 'product-modal';
        modal.className = 'fixed inset-0 z-50 hidden';
        modal.innerHTML = `
            <div class="fixed inset-0 bg-black bg-opacity-50" onclick="reviewPR.closeModal()"></div>
            <div class="fixed inset-0 flex items-center justify-center p-4">
                <div class="bg-white rounded-xl max-w-4xl w-full max-h-screen overflow-y-auto">
                    <div class="p-6">
                        <div class="flex justify-between items-center mb-6">
                            <h2 class="modal-title text-2xl font-bold text-gray-900"></h2>
                            <button onclick="reviewPR.closeModal()" class="text-gray-400 hover:text-gray-600">
                                <i class="fas fa-times text-xl"></i>
                            </button>
                        </div>
                        <div class="modal-body"></div>
                    </div>
                </div>
            </div>
        `;
        return modal;
    }

    closeModal() {
        const modal = document.getElementById('product-modal');
        if (modal) {
            modal.classList.add('hidden');
            document.body.style.overflow = 'auto';
        }
    }

    orderProduct(productId) {
        const product = this.products.find(p => String(p.id) === String(productId));
        if (!product) {
            this.showNotification('Product not found', 'error');
            return;
        }

        // Close any open modals
        this.closeModal();
        
        // Show order form or redirect to checkout
        this.showOrderForm(product);
    }

    showOrderForm(product) {
        // Create a simple order form
        const orderData = {
            productId: product.id,
            productName: product.name,
            price: product.price
        };

        // For now, just show a notification
        this.showNotification(`Order initiated for ${product.name}. Redirecting to checkout...`, 'success');
        
        // Simulate redirect to checkout
        setTimeout(() => {
            // You can implement actual checkout logic here
            console.log('Redirecting to checkout with:', orderData);
        }, 1500);
    }

    handleQuickOrder(formData) {
        const selectedProductId = formData.get('platform');
        const platformSelect = document.getElementById('quick-platform');
        
        if (!selectedProductId) {
            this.showNotification('Please select a service first', 'error');
            return;
        }

        const selectedOption = platformSelect.selectedOptions[0];
        
        // Handle temporary options
        if (selectedOption.hasAttribute('data-price')) {
            const price = selectedOption.getAttribute('data-price');
            const serviceName = selectedOption.textContent.split(' - $')[0];
            
            this.showNotification(`Quick order initiated for ${serviceName} - $${parseFloat(price).toFixed(2)}`, 'success');
            
            setTimeout(() => {
                this.showNotification('Redirecting to full product catalog...', 'info');
                // Scroll to products section
                document.getElementById('pricing')?.scrollIntoView({ behavior: 'smooth' });
            }, 1500);
            
            console.log('Quick order (temporary):', { 
                serviceName: serviceName, 
                price: price 
            });
            return;
        }

        // Handle real products
        const product = this.products.find(p => p.id == selectedProductId);
        
        if (!product) {
            this.showNotification('Selected service not found', 'error');
            return;
        }

        // Show success message
        this.showNotification(`Quick order initiated for ${product.name} - $${parseFloat(product.price).toFixed(2)}`, 'success');
        
        // Redirect to product order or show order modal
        setTimeout(() => {
            this.orderProduct(selectedProductId);
        }, 1500);
        
        console.log('Quick order (real product):', { 
            productId: selectedProductId, 
            productName: product.name, 
            price: product.price 
        });
    }

    showNotification(message, type = 'info') {
        // Create notification element
        const notification = document.createElement('div');
        notification.className = `fixed top-4 right-4 z-50 p-4 rounded-lg shadow-lg max-w-sm transform transition-all duration-300 translate-x-full`;
        
        const bgColor = {
            success: 'bg-green-500',
            error: 'bg-red-500',
            warning: 'bg-yellow-500',
            info: 'bg-blue-500'
        }[type] || 'bg-blue-500';
        
        notification.classList.add(bgColor);
        notification.innerHTML = `
            <div class="flex items-center text-white">
                <i class="fas fa-${type === 'success' ? 'check' : type === 'error' ? 'exclamation-triangle' : 'info'}-circle mr-2"></i>
                <span>${message}</span>
                <button onclick="this.parentElement.parentElement.remove()" class="ml-4 text-white hover:text-gray-200">
                    <i class="fas fa-times"></i>
                </button>
            </div>
        `;
        
        document.body.appendChild(notification);
        
        // Animate in
        setTimeout(() => {
            notification.classList.remove('translate-x-full');
        }, 100);
        
        // Auto remove after 5 seconds
        setTimeout(() => {
            notification.classList.add('translate-x-full');
            setTimeout(() => {
                if (notification.parentElement) {
                    notification.remove();
                }
            }, 300);
        }, 5000);
    }

    logError(message, error = null) {
        const errorInfo = {
            message,
            error: error ? error.toString() : null,
            timestamp: new Date().toISOString(),
            url: window.location.href,
            userAgent: navigator.userAgent
        };
        
        this.performanceMetrics.errors.push(errorInfo);
        console.error('ReviewPR Error:', errorInfo);
        
        // In production, you might want to send this to an error tracking service
        // this.sendErrorToService(errorInfo);
    }

    getPerformanceMetrics() {
        return {
            ...this.performanceMetrics,
            totalTime: performance.now() - this.performanceMetrics.startTime,
            memoryUsage: performance.memory ? {
                used: Math.round(performance.memory.usedJSHeapSize / 1048576),
                total: Math.round(performance.memory.totalJSHeapSize / 1048576),
                limit: Math.round(performance.memory.jsHeapSizeLimit / 1048576)
            } : null
        };
    }

    // Initialize lazy loading for images
    initLazyLoading() {
        const images = document.querySelectorAll('img[loading="lazy"]');
        
        if ('IntersectionObserver' in window) {
            const imageObserver = new IntersectionObserver((entries, observer) => {
                entries.forEach(entry => {
                    if (entry.isIntersecting) {
                        const img = entry.target;
                        if (img.dataset.src) {
                            img.src = img.dataset.src;
                            img.removeAttribute('data-src');
                        }
                        observer.unobserve(img);
                    }
                });
            });

            images.forEach(img => imageObserver.observe(img));
        }
    }

    // Debounce function for performance optimization
    debounce(func, wait) {
        let timeout;
        return function executedFunction(...args) {
            const later = () => {
                clearTimeout(timeout);
                func(...args);
            };
            clearTimeout(timeout);
            timeout = setTimeout(later, wait);
        };
    }
}

// Global function for read more toggle (accessible from onclick attributes)
window.toggleDescription = function(button) {
    const card = button.closest('.product-card');
    const shortDesc = card.querySelector('.description-short');
    const fullDesc = card.querySelector('.description-full');
    const icon = button.querySelector('i');
    const span = button.querySelector('span');

    if (shortDesc.style.display === 'block') {
        // Expanding to show full description
        shortDesc.style.display = 'none';
        fullDesc.style.display = 'block';
        span.textContent = 'Show Less';
        icon.style.transform = 'rotate(180deg)';
        button.classList.remove('bg-blue-50', 'border-blue-200', 'text-blue-600', 'hover:text-blue-800');
        button.classList.add('bg-gray-50', 'border-gray-300', 'text-gray-600', 'hover:text-gray-800');
    } else {
        // Collapsing to show short description
        shortDesc.style.display = 'block';
        fullDesc.style.display = 'none';
        span.textContent = 'Read More';
        icon.style.transform = 'rotate(0deg)';
        button.classList.remove('bg-gray-50', 'border-gray-300', 'text-gray-600', 'hover:text-gray-800');
        button.classList.add('bg-blue-50', 'border-blue-200', 'text-blue-600', 'hover:text-blue-800');
    }
};

// Initialize the application
const reviewPR = new ReviewPR();

// Export for global access
window.reviewPR = reviewPR;
